<?php
session_start();
require 'db.php';
require_once __DIR__ . '/includes/payment_gateway.php';

if (!isset($_SESSION['usuario_id'])) {
    header('Location: index.php');
    exit;
}

$usuario_id = $_SESSION['usuario_id'];
$valor = floatval($_POST['valor'] ?? 0);
// CPF/CNPJ opcional vindo do formulário para compor o customer.document
$taxIdPost = preg_replace('/\D/', '', (string)($_POST['cpf_cnpj'] ?? ''));

// ===== validação de limites =====
$stmt = $pdo->query("SELECT min_deposito, max_deposito FROM configuracoes LIMIT 1");
$config = $stmt->fetch(PDO::FETCH_ASSOC);
$min_deposito = floatval($config['min_deposito']);
$max_deposito = floatval($config['max_deposito']);

if ($valor < $min_deposito || $valor > $max_deposito) {
    echo "⚠️ Valor inválido! O valor deve ser entre R$ " . number_format($min_deposito, 2, ',', '.') . " e R$ " . number_format($max_deposito, 2, ',', '.');
    exit;
}

// ===== busca dados do perfil para enriquecer o payload (LotusPay customer) =====
$stmtUser = $pdo->prepare("SELECT nome, email, telefone FROM usuarios WHERE id = ?");
$stmtUser->execute([$usuario_id]);
$perfil = $stmtUser->fetch(PDO::FETCH_ASSOC) ?: ['nome' => '', 'email' => '', 'telefone' => ''];
$nomePerfil = trim($perfil['nome'] ?? '');
$emailPerfil = trim($perfil['email'] ?? '');
$telefonePerfil = preg_replace('/\D/', '', (string)($perfil['telefone'] ?? ''));

// ===== cria cobrança PIX via fábrica segura =====
$gateway = PaymentGatewayFactory::getGateway($pdo);
if (!$gateway->isAvailable() || $gateway->getName() === 'null') {
    echo "❌ Gateway PIX indisponível no momento.";
    exit;
}

$result = $gateway->generatePixCharge($usuario_id, $valor, [
    'name' => $nomePerfil,
    'email' => $emailPerfil,
    'phone' => $telefonePerfil,
    'tax_id' => $taxIdPost,
]);

if (!$result['ok']) {
    echo "❌ Erro ao gerar QR Code PIX. " . htmlspecialchars($result['message'] ?? '');
    exit;
}

$qrcode = $result['qrcode'];
$transaction_id = $result['transaction_id'] ?? null;
$external_id = $result['external_id'];

// ===== salva transação =====
$telefone = $telefonePerfil;

$stmt = $pdo->prepare("INSERT INTO transacoes_pix 
    (usuario_id, telefone, valor, external_id, transaction_id, status, conta_recebedora, criado_em) 
    VALUES (?, ?, ?, ?, ?, ?, ?, NOW())");

$stmt->execute([$usuario_id, $telefone, $valor, $external_id, $transaction_id, 'pendente', 'FIXA']);


?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
  <meta charset="UTF-8">
  <title>PIX Gerado</title>
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;600;700&display=swap" rel="stylesheet">
  <link rel="stylesheet" href="/css/xu2fL9a0.css">
</head>
<body>
  <div class="container">
    <div class="title">PIX gerado: R$ <?= number_format($valor, 2, ',', '.') ?></div>
    <div class="subtitle">Escaneie o QR Code ou copie o código abaixo:</div>
    <div class="qrcode">
      <img src="https://api.qrserver.com/v1/create-qr-code/?size=250x250&data=<?= urlencode($qrcode) ?>" alt="QR Code PIX">
    </div>
    <textarea id="codigoPix" rows="4" readonly><?= htmlspecialchars($qrcode) ?></textarea>
    <button class="btn-copiar" onclick="copiarCodigo()">Copiar código PIX</button>
    <div class="msg-copiado" id="mensagemCopiado">Codigo copiado com sucesso!</div>

    <div class="tutorial">
      <h3>Como pagar com PIX:</h3>
      <div class="passo"><span>1</span>Abra seu aplicativo do banco</div>
      <div class="passo"><span>2</span>Escolha a opção <strong>PIX</strong></div>
      <div class="passo"><span>3</span>Selecione <strong>Pix Copia e Cola</strong></div>
      <div class="passo"><span>4</span>Cole o código que você copiou acima</div>
      <div class="passo"><span>5</span>Confirme o valor e finalize o pagamento</div>
    </div>

    <div id="statusPagamento" style="text-align:center; margin-top: 15px; color: #00e880;"></div>
  </div>

  <script>
    function copiarCodigo() {
      const textarea = document.getElementById("codigoPix");
      textarea.select();
      document.execCommand("copy");
      const msg = document.getElementById("mensagemCopiado");
      msg.style.display = "block";
      setTimeout(() => msg.style.display = "none", 3000);
    }
  </script>

  <script>
    const externalId = "<?= $external_id ?>";
    const statusMsg = document.getElementById("statusPagamento");
    function verificarStatus() {
      fetch(`status_transacao.php?external_id=${externalId}`)
        .then(res => res.json())
        .then(data => {
          if (data.status === 'aprovado') {
            statusMsg.innerHTML = `✅ Pagamento confirmado!<br>Seu saldo atual é R$ ${data.saldo}`;
            fbq('track', 'Purchase', { value: <?= $valor ?>, currency: 'BRL' });
            setTimeout(() => window.location.href = 'index.php', 2000);
          } else if (data.status === 'pendente') {
            statusMsg.innerHTML = '⏳ Aguardando confirmação do pagamento...';
          } else {
            statusMsg.innerHTML = '⚠️ Status desconhecido.';
          }
        })
        .catch(() => statusMsg.innerHTML = '❌ Erro ao verificar status.');
    }
    verificarStatus();
    setInterval(verificarStatus, 5000);
  </script>
</body>
</html>
