<?php
// Centralized Payment Gateway Factory and Implementations for Pix
// Provides a tamper-resistant single entry point to create Pix charges

final class PaymentGatewayFactory {
    public static function getGateway(PDO $pdo): PaymentGatewayInterface {
        try {
            $stmt = $pdo->query("SELECT nome, client_id, client_secret, callback_url, ativo FROM gateways WHERE ativo = 1 LIMIT 1");
            $gw = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$gw) {
                return new NullGateway('Nenhum gateway ativo configurado.');
            }

            $nome = strtolower(trim($gw['nome'] ?? ''));
            $clientId = trim($gw['client_id'] ?? '');
            $clientSecret = trim($gw['client_secret'] ?? '');
            $callbackUrl = trim($gw['callback_url'] ?? '');

            if ($clientId === '' || $clientSecret === '' || $callbackUrl === '') {
                return new NullGateway('Gateway configurado de forma incompleta.');
            }

            // Force LotusPay only. Use client_secret as API token.
            return new LotusPayGateway($clientSecret, $callbackUrl);
        } catch (Throwable $e) {
            return new NullGateway('Falha ao carregar gateway: ' . $e->getMessage());
        }
    }
}

interface PaymentGatewayInterface {
    public function getName(): string;
    public function isAvailable(): bool;
    public function getCallbackUrl(): string;

    /**
     * Create a Pix charge
     * @return array{ok:bool, message?:string, qrcode?:string, transaction_id?:string|null, external_id?:string}
     */
    public function generatePixCharge(int $userId, float $amount, array $options = []): array;

    /**
     * Create a Pix payout (cash-out)
     * Options: key, key_type, name, tax_id
     * @return array{ok:bool, message?:string, external_id?:string, transaction_id?:string|null}
     */
    public function generatePixPayout(int $userId, float $amount, array $options = []): array;
}

final class NullGateway implements PaymentGatewayInterface {
    private string $reason;
    public function __construct(string $reason = 'Gateway indisponível') {
        $this->reason = $reason;
    }
    public function getName(): string { return 'null'; }
    public function isAvailable(): bool { return false; }
    public function getCallbackUrl(): string { return ''; }
    public function generatePixCharge(int $userId, float $amount, array $options = []): array {
        return ['ok' => false, 'message' => $this->reason];
    }
    public function generatePixPayout(int $userId, float $amount, array $options = []): array {
        return ['ok' => false, 'message' => $this->reason];
    }
}

final class LotusPayGateway implements PaymentGatewayInterface {
    private string $apiToken;
    private string $callbackUrlDeposit;
    private string $callbackUrlPayout;

    public function __construct(string $apiToken, string $callbackUrlDeposit) {
        $this->apiToken = $apiToken;
        $this->callbackUrlDeposit = $callbackUrlDeposit;
        // Derive payout callback: replace file name if present
        $base = rtrim($callbackUrlDeposit, '/');
        if (str_ends_with($base, '/webhook-pix.php')) {
            $this->callbackUrlPayout = substr($base, 0, -strlen('/webhook-pix.php')) . '/webhook_saque.php';
        } else {
            $this->callbackUrlPayout = $base . '/webhook_saque.php';
        }
    }

    public function getName(): string { return 'lotuspay'; }
    public function isAvailable(): bool { return $this->apiToken !== '' && $this->callbackUrlDeposit !== ''; }
    public function getCallbackUrl(): string { return $this->callbackUrlDeposit; }

    public function generatePixCharge(int $userId, float $amount, array $options = []): array {
        if ($amount <= 0) {
            return ['ok' => false, 'message' => 'Valor inválido'];
        }

        $externalId = $options['external_id'] ?? md5(uniqid("user{$userId}_", true));

        // Payload para LotusPay cashIn (inclui customer opcional e callbackUrl)
        $payload = [
            'amount' => $amount,
            'externalId' => $externalId,
            'callbackUrl' => $this->callbackUrlDeposit,
            'postbackUrl' => $this->callbackUrlDeposit,
        ];

        $name = trim((string)($options['name'] ?? ''));
        $email = trim((string)($options['email'] ?? ''));
        $phone = preg_replace('/\D/', '', (string)($options['phone'] ?? ''));
        $taxId = preg_replace('/\D/', '', (string)($options['tax_id'] ?? ''));

        // Fallbacks obrigatórios
        if ($name === '') { $name = "Usuario {$userId}"; }
        if ($email === '') { $email = "usuario{$userId}@example.com"; }
        if ($phone === '') { $phone = '00000000000'; }
        if ($taxId === '') { $taxId = '00000000000'; }

        $docType = strlen($taxId) === 14 ? 'cnpj' : 'cpf';
        $payload['customer'] = [
            'document' => [ 'type' => $docType, 'number' => $taxId ],
            'name' => $name,
            'email' => $email,
            'phone' => $phone,
        ];

        $resp = $this->postJson('https://api.lotuspay.me/api/v1/cashin', $payload, [
            'Lotuspay-Auth: ' . $this->apiToken,
            'Content-Type: application/json',
            'Accept: application/json',
        ]);

        if (!$resp['ok']) {
            return ['ok' => false, 'message' => 'Erro ao chamar API LotusPay'];
        }

        $body = $resp['body'];
        // Accept both 'qrCode' and 'qrcode'
        $qr = $body['qrCode'] ?? ($body['qrcode'] ?? null);
        if (!$qr) {
            return ['ok' => false, 'message' => 'Resposta inválida da LotusPay'];
        }
        $txId = $body['publicId'] ?? ($body['transactionId'] ?? ($body['id'] ?? null));

        return [
            'ok' => true,
            'qrcode' => $qr,
            'transaction_id' => $txId,
            'external_id' => $externalId,
        ];
    }

    public function generatePixPayout(int $userId, float $amount, array $options = []): array {
        if ($amount <= 0) {
            return ['ok' => false, 'message' => 'Valor inválido'];
        }

        $key = trim((string)($options['key'] ?? ''));
        $keyType = strtoupper(trim((string)($options['key_type'] ?? '')));
        $name = trim((string)($options['name'] ?? ''));
        $taxId = preg_replace('/\D/', '', (string)($options['tax_id'] ?? ''));
        if ($key === '' || $keyType === '' || $name === '' || $taxId === '') {
            return ['ok' => false, 'message' => 'Dados do recebedor incompletos'];
        }

        $externalId = $options['external_id'] ?? md5(uniqid("cashout_user{$userId}_", true));

        // LotusPay cashOut (payload conforme exemplo informado)
        $docType = strlen($taxId) === 11 ? 'cpf' : (strlen($taxId) === 14 ? 'cnpj' : 'cpf');
        $payload = [
            'amount' => $amount,
            'externalId' => $externalId,
            'callbackUrl' => $this->callbackUrlPayout,
            'pixKeyType' => strtolower($keyType),
            'pixKey' => $key,
            'customer' => [
                'document' => [
                    'type' => $docType,
                    'number' => $taxId,
                ],
                'name' => $name,
                'email' => $options['email'] ?? null,
                'phone' => $options['phone'] ?? null,
            ],
        ];

        $resp = $this->postJson('https://api.lotuspay.me/api/v1/cashout', $payload, [
            'Lotuspay-Auth: ' . $this->apiToken,
            'Content-Type: application/json',
            'Accept: application/json',
        ]);

        if (!$resp['ok']) {
            return ['ok' => false, 'message' => 'Erro ao chamar API LotusPay cashout'];
        }
        $body = $resp['body'];
        $txId = $body['publicId'] ?? ($body['transactionId'] ?? ($body['id'] ?? null));
        return [
            'ok' => true,
            'external_id' => $externalId,
            'transaction_id' => $txId,
        ];
    }

    private function postJson(string $url, array $payload, array $headers): array {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        $response = curl_exec($ch);
        $err = curl_error($ch);
        curl_close($ch);
        if ($err) {
            return ['ok' => false, 'error' => $err, 'body' => null];
        }
        $body = json_decode($response, true);
        return ['ok' => true, 'body' => $body];
    }
}
