<?php
require 'db.php';

// ==================== Headers ====================
header("Content-Type: application/json; charset=utf-8");
header("Access-Control-Allow-Origin: *"); // Apenas para teste; em produção restrinja o domínio
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

$logFile = "log_webhook_pixup.txt";

// Função de log
function writeLog($message) {
    global $logFile;
    file_put_contents($logFile, date('[Y-m-d H:i:s] ') . $message . PHP_EOL, FILE_APPEND);
}

// ==================== Bloquear métodos inválidos ====================
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    writeLog("Acesso inválido: método " . $_SERVER['REQUEST_METHOD']);
    http_response_code(405);
    echo json_encode(['error' => 'Use POST com JSON']);
    exit;
}

// ==================== Lê corpo JSON ====================
$raw = file_get_contents('php://input');
writeLog("Recebido: " . $raw);

// ==================== Decodifica JSON ====================
$data = json_decode($raw, true);

if (!$data || !is_array($data)) {
    writeLog("Erro: Dados inválidos ou vazios");
    http_response_code(400);
    echo json_encode(['error' => 'Dados inválidos ou vazios']);
    exit;
}

// ==================== Mapeia campos (compat: PixUp e LotusPay) ====================
$external_id_webhook = $data['external_id'] ?? ($data['externalId'] ?? null);
$transaction_id      = $data['transactionId'] ?? ($data['publicId'] ?? null);
$status_webhook_raw  = $data['status'] ?? null;
$valor_pago          = isset($data['amount']) ? floatval($data['amount']) : 0;
$data_aprovado       = $data['dateApproval'] ?? ($data['paidAt'] ?? date('Y-m-d H:i:s'));

if (!$external_id_webhook || !$status_webhook_raw) {
    writeLog("Erro: Dados incompletos");
    http_response_code(400);
    echo json_encode(['error' => 'Dados incompletos']);
    exit;
}

// ==================== Dados do webhook ====================
$transaction_id      = trim((string)$transaction_id);
$external_id_webhook = trim((string)$external_id_webhook);
$status_webhook      = strtoupper(trim((string)$status_webhook_raw));
$valor_pago          = floatval($valor_pago);
$data_aprovado       = $data_aprovado;

writeLog("Status recebido: {$status_webhook}, Valor: {$valor_pago}, External ID: {$external_id_webhook}");

// ==================== Status de sucesso ====================
$STATUS_OK = ['PAID','APPROVED','CONFIRMED','SUCCEEDED'];

if (in_array($status_webhook, $STATUS_OK, true)) {

    // Busca transação pendente
    $stmt = $pdo->prepare("SELECT * FROM transacoes_pix WHERE external_id = ? AND LOWER(status) = 'pendente' LIMIT 1");
    $stmt->execute([$external_id_webhook]);
    $transacao = $stmt->fetch();

    if ($transacao) {
        writeLog("Transação pendente encontrada: ID {$transacao['id']}");

        // Atualiza status para aprovado
        $stmtUp = $pdo->prepare("UPDATE transacoes_pix SET status = 'aprovado', valor = ?, transaction_id = ? WHERE id = ?");
        $stmtUp->execute([$valor_pago, $transaction_id, $transacao['id']]);

        // Busca configurações
        $config = $pdo->query("SELECT bonus_deposito, valor_comissao, valor_comissao_n2, rollover_multiplicador FROM configuracoes LIMIT 1")->fetch();
        $bonusPercent          = floatval($config['bonus_deposito']) / 100;
        $percentualComissaoN1  = floatval($config['valor_comissao']) / 100;
        $percentualComissaoN2  = floatval($config['valor_comissao_n2']) / 100;
        $rolloverMultiplicador = floatval($config['rollover_multiplicador']) ?: 2;

        $bonusValor = $valor_pago * $bonusPercent;
        $valorFinal = $valor_pago + $bonusValor;

        writeLog("Valor final (com bônus): {$valorFinal}, Bônus aplicado: {$bonusValor}");

        // Atualiza saldo do usuário
        $stmtSaldo = $pdo->prepare("UPDATE usuarios SET saldo = saldo + ? WHERE id = ?");
        $stmtSaldo->execute([$valorFinal, $transacao['usuario_id']]);

        // Cria rollover se não existir
        $stmtRol = $pdo->prepare("SELECT COUNT(*) FROM rollover WHERE usuario_id = ? AND valor_deposito = ? AND finalizado = 0");
        $stmtRol->execute([$transacao['usuario_id'], $valor_pago]);
        if (!$stmtRol->fetchColumn()) {
            $valorRollover = $valor_pago * $rolloverMultiplicador;
            $stmtIR = $pdo->prepare("INSERT INTO rollover (usuario_id, valor_deposito, valor_necessario, valor_acumulado, finalizado, criado_em) VALUES (?, ?, ?, 0, 0, NOW())");
            $stmtIR->execute([$transacao['usuario_id'], $valor_pago, $valorRollover]);
            writeLog("Rollover criado: {$valorRollover}");
        }

        // ======= Comissão Nível 1 =======
        $stmtInd = $pdo->prepare("SELECT indicado_por FROM usuarios WHERE id = ?");
        $stmtInd->execute([$transacao['usuario_id']]);
        $indicadorNivel1 = $stmtInd->fetchColumn();

        if ($indicadorNivel1) {
            $valorComissaoN1 = $valor_pago * $percentualComissaoN1;
            $stmtCI1 = $pdo->prepare("UPDATE usuarios SET comissao = comissao + ? WHERE id = ?");
            $stmtCI1->execute([$valorComissaoN1, $indicadorNivel1]);

            $stmtIC1 = $pdo->prepare("INSERT INTO comissoes (usuario_id, indicado_id, transacao_id, valor, nivel) VALUES (?, ?, ?, ?, 1)");
            $stmtIC1->execute([$indicadorNivel1, $transacao['usuario_id'], $transacao['id'], $valorComissaoN1]);

            writeLog("Comissão N1 registrada: {$valorComissaoN1} para usuário {$indicadorNivel1}");

            // ======= Comissão Nível 2 =======
            $stmtInd2 = $pdo->prepare("SELECT indicado_por FROM usuarios WHERE id = ?");
            $stmtInd2->execute([$indicadorNivel1]);
            $indicadorNivel2 = $stmtInd2->fetchColumn();

            if ($indicadorNivel2) {
                $valorComissaoN2 = $valor_pago * $percentualComissaoN2;
                $stmtCI2 = $pdo->prepare("UPDATE usuarios SET comissao = comissao + ? WHERE id = ?");
                $stmtCI2->execute([$valorComissaoN2, $indicadorNivel2]);

                $stmtIC2 = $pdo->prepare("INSERT INTO comissoes (usuario_id, indicado_id, transacao_id, valor, nivel) VALUES (?, ?, ?, ?, 2)");
                $stmtIC2->execute([$indicadorNivel2, $transacao['usuario_id'], $transacao['id'], $valorComissaoN2]);

                writeLog("Comissão N2 registrada: {$valorComissaoN2} para usuário {$indicadorNivel2}");
            }
        }

        http_response_code(200);
        echo json_encode(['message' => 'Depósito aprovado e processado']);
        exit;

    } else {
        writeLog("Transação não encontrada ou já processada: {$external_id_webhook}");
        http_response_code(200);
        echo json_encode(['message' => 'Transação não encontrada ou já processada']);
        exit;
    }
}

// Outros status (PENDING, FAILED, etc.)
writeLog("Status {$status_webhook} ignorado");
http_response_code(200);
echo json_encode(['message' => "Status {$status_webhook} ignorado"]);
exit;
